import os
import json
import requests
from flask import Blueprint, redirect, request, url_for, current_app, flash
from flask_login import login_user, logout_user, login_required
from oauthlib.oauth2 import WebApplicationClient
from models import db, User
from datetime import datetime

# Allow insecure transport for local development (testing over HTTP)
os.environ['OAUTHLIB_INSECURE_TRANSPORT'] = '1'

auth_bp = Blueprint('auth', __name__)

client = None

def get_google_provider_cfg():
    return requests.get(current_app.config['GOOGLE_DISCOVERY_URL']).json()

@auth_bp.route("/google")
def google_login():
    # Find out what URL to hit for Google login
    google_provider_cfg = get_google_provider_cfg()
    authorization_endpoint = google_provider_cfg["authorization_endpoint"]

    # Use library to construct the request for Google login and provide
    # scopes that let you retrieve user's profile from Google
    client = WebApplicationClient(current_app.config['GOOGLE_CLIENT_ID'])
    request_uri = client.prepare_request_uri(
        authorization_endpoint,
        redirect_uri=url_for("auth.google_callback", _external=True),
        scope=["openid", "email", "profile"],
    )
    return redirect(request_uri)

@auth_bp.route("/google/callback")
def google_callback():
    # Get authorization code Google sent back to you
    code = request.args.get("code")

    # Find out what URL to hit to get tokens that allow you to ask for
    # things on behalf of a user
    google_provider_cfg = get_google_provider_cfg()
    token_endpoint = google_provider_cfg["token_endpoint"]

    # Prepare and send a request to get tokens! Yay tokens!
    client = WebApplicationClient(current_app.config['GOOGLE_CLIENT_ID'])
    token_url, headers, body = client.prepare_token_request(
        token_endpoint,
        authorization_response=request.url,
        redirect_url=url_for("auth.google_callback", _external=True),
        code=code
    )
    token_response = requests.post(
        token_url,
        headers=headers,
        data=body,
        auth=(current_app.config['GOOGLE_CLIENT_ID'], current_app.config['GOOGLE_CLIENT_SECRET']),
    )

    # Parse the tokens!
    client.parse_request_body_response(json.dumps(token_response.json()))

    # Now that you have tokens (yay) let's find and hit the URL
    # from Google that gives you the user's profile information,
    # including their Google profile image and email
    userinfo_endpoint = google_provider_cfg["userinfo_endpoint"]
    uri, headers, body = client.add_token(userinfo_endpoint)
    userinfo_response = requests.get(uri, headers=headers, data=body)

    # You want to make sure their email is verified.
    # The user authenticated with Google, so trust Google.
    if userinfo_response.json().get("email_verified"):
        unique_id = userinfo_response.json()["sub"]
        users_email = userinfo_response.json()["email"]
        picture = userinfo_response.json()["picture"]
        users_name = userinfo_response.json()["given_name"]
    else:
        return "User email not available or not verified by Google.", 400

    # Create a user in your database with the information provided
    # by Google
    user = User.query.filter_by(google_id=unique_id).first()
    
    if not user:
        # Check if user with same email exists
        user = User.query.filter_by(email=users_email).first()
        if user:
            # Update existing user with Google ID
            user.google_id = unique_id
        else:
            # Create new user
            user = User(
                google_id=unique_id,
                username=users_email,
                email=users_email,
                name=users_name,
                picture_url=picture
            )
            db.session.add(user)

    # Update user details
    user.name = users_name
    user.picture_url = picture
    user.last_login = datetime.utcnow()
    db.session.commit()

    # Log in the user (Remember Me enabled by default for social login)
    login_user(user, remember=True)

    # Send user back to homepage
    return redirect(url_for("index"))
