document.addEventListener('DOMContentLoaded', function() {
    const fileInput = document.getElementById('fileInput');
    const fileInputLabel = document.querySelector('.file-input-label');
    const convertBtn = document.getElementById('convertBtn');
    const uploadForm = document.getElementById('uploadForm');
    const processingMessage = document.getElementById('processingMessage'); // 👈 Added reference

    // File input change handler
    fileInput.addEventListener('change', function(e) {
        if (this.files && this.files[0]) {
            const file = this.files[0];
            const fileText = fileInputLabel.querySelector('.file-text');
            const fileSize = fileInputLabel.querySelector('.file-size');

            fileText.textContent = file.name;
            fileSize.textContent = formatFileSize(file.size);

            fileInputLabel.style.borderColor = '#10b981';
            fileInputLabel.style.background = '#f0fdf4';
        }
    });

    // Form submission handler
    uploadForm.addEventListener('submit', function(e) {
        const file = fileInput.files[0];
        const conversionType = document.querySelector('input[name="conversion_type"]:checked');

        if (!file) {
            e.preventDefault();
            showAlert('Please select a PDF file', 'error');
            return;
        }

        if (!conversionType) {
            e.preventDefault();
            showAlert('Please select a conversion type', 'error');
            return;
        }

        // Show loading state
        convertBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Converting...';
        convertBtn.disabled = true;

        // 👇 NEW: Show processing message
        if (processingMessage) {
            processingMessage.style.display = 'flex';
        }
    });

    // Format file size
    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    // Show alert message
    function showAlert(message, type) {
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type}`;
        alertDiv.textContent = message;

        const container = document.querySelector('.container');
        container.insertBefore(alertDiv, container.firstChild);

        setTimeout(() => {
            alertDiv.remove();
        }, 5000);
    }

    // Drag and drop functionality
    fileInputLabel.addEventListener('dragover', function(e) {
        e.preventDefault();
        this.style.borderColor = '#4f46e5';
        this.style.background = '#f0f4ff';
    });

    fileInputLabel.addEventListener('dragleave', function(e) {
        e.preventDefault();
        if (!this.contains(e.relatedTarget)) {
            this.style.borderColor = '#4f46e5';
            this.style.background = '#f8faff';
        }
    });

    fileInputLabel.addEventListener('drop', function(e) {
        e.preventDefault();
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            fileInput.files = files;
            fileInput.dispatchEvent(new Event('change'));
        }
    });
});
document.addEventListener('DOMContentLoaded', function() {
    const customRadio = document.getElementById('customRadio');
    const customPromptSection = document.getElementById('customPromptSection');
    const customPromptTextarea = document.getElementById('customPrompt');
    const charCount = document.getElementById('charCount');
    const convertBtn = document.getElementById('convertBtn');
    const uploadForm = document.getElementById('uploadForm');
    const uploadCard = document.querySelector('.upload-card');
    const processingMessage = document.getElementById('processingMessage');
    const fileInput = document.getElementById('fileInput');
    const fileLabel = document.querySelector('.file-input-label');

    // Initialize character counter
    charCount.textContent = customPromptTextarea.value.length;

    // Toggle custom prompt section and update UI
    document.querySelectorAll('input[name="conversion_type"]').forEach(radio => {
        radio.addEventListener('change', function() {
            if (this.value === 'custom') {
                // Show custom section
                customPromptSection.style.display = 'block';
                customPromptTextarea.required = true;

                // Update button text
                convertBtn.innerHTML = '<i class="fas fa-wand-magic-sparkles"></i> Convert with Custom Rules';

                // Add custom mode class to form
                uploadCard.classList.add('custom-mode');
                convertBtn.classList.add('custom-mode');

                // Update character counter
                charCount.textContent = customPromptTextarea.value.length;

                // Auto-focus the textarea
                setTimeout(() => {
                    customPromptTextarea.focus();
                }, 300);
            } else {
                // Hide custom section
                customPromptSection.style.display = 'none';
                customPromptTextarea.required = false;

                // Restore default button
                convertBtn.innerHTML = '<i class="fas fa-magic"></i> Convert to Excel';

                // Remove custom mode classes
                uploadCard.classList.remove('custom-mode');
                convertBtn.classList.remove('custom-mode');
            }
        });
    });

    // Character counter update
    customPromptTextarea.addEventListener('input', function() {
        const count = this.value.length;
        charCount.textContent = count;

        // Visual feedback for character count
        if (count < 20) {
            charCount.style.color = '#dc2626';
        } else if (count < 50) {
            charCount.style.color = '#f59e0b';
        } else {
            charCount.style.color = '#10b981';
        }
    });

    // Example buttons functionality
    document.querySelectorAll('.example-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            customPromptTextarea.value = this.dataset.prompt;
            customPromptTextarea.dispatchEvent(new Event('input'));
            customPromptTextarea.focus();

            // Smooth scroll to textarea
            customPromptTextarea.scrollIntoView({
                behavior: 'smooth',
                block: 'center'
            });

            // Visual feedback
            this.style.animation = 'pulse 0.5s ease';
            setTimeout(() => {
                this.style.animation = '';
            }, 500);
        });
    });

    // Form validation
    uploadForm.addEventListener('submit', function(e) {
        const selectedConversion = document.querySelector('input[name="conversion_type"]:checked');

        if (!selectedConversion) {
            e.preventDefault();
            showError('Please select a conversion type');
            return;
        }

        if (selectedConversion.value === 'custom') {
            const prompt = customPromptTextarea.value.trim();

            // Validate custom prompt
            if (!prompt) {
                e.preventDefault();
                showError('Please enter custom conversion instructions');
                customPromptTextarea.focus();
                return;
            }

            if (prompt.length < 10) {
                e.preventDefault();
                showError('Please provide more detailed instructions (at least 10 characters)');
                customPromptTextarea.focus();
                return;
            }
        }

        // Check if file is selected
        if (!fileInput.files.length) {
            e.preventDefault();
            showError('Please select a PDF file to convert');
            return;
        }

        // Show processing message
        processingMessage.style.display = 'flex';
        convertBtn.disabled = true;

        // Update button based on conversion type
        if (selectedConversion.value === 'custom') {
            convertBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing with AI...';
        } else {
            convertBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
        }
    });

    // File input styling
    fileInput.addEventListener('change', function() {
        if (this.files.length > 0) {
            const fileName = this.files[0].name;
            const fileSize = (this.files[0].size / 1024 / 1024).toFixed(2);

            // Update file label
            fileLabel.innerHTML = `
                <i class="fas fa-file-pdf"></i>
                <span class="file-text">${fileName}</span>
                <span class="file-size">${fileSize} MB</span>
            `;

            // Add visual feedback
            fileLabel.classList.add('file-selected');

            // Remove any existing file errors
            removeFileError();
        } else {
            // Reset to default
            fileLabel.innerHTML = `
                <i class="fas fa-file-pdf"></i>
                <span class="file-text">Choose PDF File</span>
                <span class="file-size">Max 16MB</span>
            `;
            fileLabel.classList.remove('file-selected');
        }
    });

    // Helper function for errors
    function showError(message) {
        // Remove any existing error
        removeError();

        const errorDiv = document.createElement('div');
        errorDiv.className = 'error-message';
        errorDiv.style.cssText = `
            animation: slideIn 0.3s ease;
        `;
        errorDiv.innerHTML = `
            <i class="fas fa-exclamation-circle"></i>
            <div class="error-details">
                <strong>Action Required</strong>
                <span>${message}</span>
            </div>
        `;

        // Insert after the file input container
        fileInput.parentNode.parentNode.insertBefore(errorDiv, fileInput.parentNode.nextSibling);

        // Auto-remove after 5 seconds
        setTimeout(removeError, 5000);
    }

    // Helper to remove errors
    function removeError() {
        const error = document.querySelector('.error-message');
        if (error) {
            error.style.animation = 'slideIn 0.3s ease reverse';
            setTimeout(() => {
                if (error.parentNode) {
                    error.parentNode.removeChild(error);
                }
            }, 300);
        }
    }

    function removeFileError() {
        const error = document.querySelector('.error-message');
        if (error && error.textContent.includes('PDF file')) {
            removeError();
        }
    }

    // Add some example prompts on page load for inspiration
    window.addEventListener('load', function() {
        // Add placeholder animation
        const placeholderExamples = [
            "Extract all financial transactions with columns: Date, Description, Amount, Category, Account",
            "Convert this invoice with columns: Invoice #, Date, Customer, Item, Quantity, Unit Price, Total",
            "Extract employee data with columns: Name, Department, Salary, Start Date, Email",
            "Extract product catalog with columns: Product ID, Product Name, Category, Price, Stock Quantity",
            "Convert bank statement with columns: Transaction Date, Description, Debit, Credit, Balance, Payee"
        ];

        let exampleIndex = 0;

        // Rotate placeholder text
        function rotatePlaceholder() {
            if (document.activeElement !== customPromptTextarea) {
                customPromptTextarea.placeholder = placeholderExamples[exampleIndex];
                exampleIndex = (exampleIndex + 1) % placeholderExamples.length;
            }
        }

        // Start rotating every 3 seconds
        let placeholderInterval = setInterval(rotatePlaceholder, 3000);

        // Stop rotating when user focuses on textarea
        customPromptTextarea.addEventListener('focus', function() {
            clearInterval(placeholderInterval);
        });

        // Resume when user leaves textarea (if empty)
        customPromptTextarea.addEventListener('blur', function() {
            if (!this.value.trim()) {
                placeholderInterval = setInterval(rotatePlaceholder, 3000);
            }
        });
    });
});